class Sprite {
    constructor({ position, imageSrc, images = null, scale = 1, framesMax = 1, offset = { x: 0, y: 0 }, isBackground = false, flip = false }) {
        this.position = position;
        this.width = 50;
        this.height = 150;
        this.scale = scale;
        this.framesMax = framesMax;
        this.frameCurrent = 0;
        this.framesElapsed = 0;
        this.framesHold = 6; // Faster animation playback
        this.offset = offset;
        this.isBackground = isBackground;
        this.flip = flip;  // If true, flip sprite horizontally
        
        // Support for individual image arrays
        if (images && Array.isArray(images)) {
            this.images = images.map(imgSrc => {
                const img = new Image();
                img.src = imgSrc;
                return img;
            });
            this.image = this.images[0];
        } else {
            // Use sprite sheet (original behavior)
            this.image = new Image();
            if (imageSrc) {
                this.image.src = imageSrc;
                // Add error handling for background images
                if (isBackground) {
                    this.image.onload = () => {
                        console.log('✅ Background image loaded:', imageSrc);
                    };
                    this.image.onerror = () => {
                        console.error('❌ Failed to load background image:', imageSrc);
                    };
                }
            }
            this.images = null;
        }
    }

    draw() {
        const TARGET_HEIGHT = 200; // Target height in pixels for both characters
        
        // Check if this is a background
        if (this.isBackground && this.image) {
            // Draw background to fill entire canvas
            if (this.image.complete && this.image.width > 0) {
                context.drawImage(
                    this.image,
                    0,
                    0,
                    canvas.width,
                    canvas.height
                )
            }
        } else if (this.images && Array.isArray(this.images) && this.images.length > 0) {
            // Draw from individual images array
            const frameIndex = Math.min(this.frameCurrent, this.images.length - 1);
            const currentImage = this.images[frameIndex];
            // Draw even if image is still loading (will show once loaded)
            if (currentImage) {
                // Check if image is loaded - use naturalWidth/Height if available, otherwise try width/height
                // naturalWidth/Height are 0 until image loads, but width/height might be set by CSS
                let imgWidth = currentImage.naturalWidth;
                let imgHeight = currentImage.naturalHeight;
                
                // If natural dimensions aren't available yet, try regular dimensions
                if (imgWidth === 0 || imgHeight === 0) {
                    imgWidth = currentImage.width;
                    imgHeight = currentImage.height;
                }
                
                // If still no dimensions, image hasn't loaded - use a default size to at least attempt drawing
                // This will fail gracefully but the image will appear once loaded
                if (imgWidth === 0 || imgHeight === 0) {
                    // Try to draw anyway - browser will handle it
                    imgWidth = 100;
                    imgHeight = 200;
                }
                
                // Always try to draw - browser will handle incomplete images
                try {
                    const heightScale = TARGET_HEIGHT / imgHeight;
                    const scaledWidth = imgWidth * heightScale;
                    const scaledHeight = TARGET_HEIGHT;
                    
                    // Check if sprite should be flipped horizontally
                    if (this.flip) {
                        context.save();
                        // Flip by scaling and translating
                        context.scale(-1, 1);
                        const drawX = -(this.position.x - this.offset.x + scaledWidth);
                        context.drawImage(
                            currentImage,
                            drawX,
                            this.position.y - this.offset.y,
                            scaledWidth,
                            scaledHeight
                        );
                        context.restore();
                    } else {
                        context.drawImage(
                            currentImage,
                            this.position.x - this.offset.x,
                            this.position.y - this.offset.y,
                            scaledWidth,
                            scaledHeight
                        );
                    }
                } catch (e) {
                    // Image might not be fully loaded yet - this is expected and will resolve when image loads
                }
            }
        } else if (this.image) {
            // Draw from sprite sheet (original behavior)
            const imgWidth = this.image.width || this.image.naturalWidth || 100;
            const imgHeight = this.image.height || this.image.naturalHeight || 200;
            
            if (imgWidth > 0 && imgHeight > 0) {
                // Calculate scale to make height exactly 200px
                const heightScale = TARGET_HEIGHT / imgHeight;
                const frameWidth = imgWidth / this.framesMax;
                const scaledFrameWidth = frameWidth * heightScale;
                const scaledHeight = TARGET_HEIGHT;
                
                // Check if sprite should be flipped horizontally
                if (this.flip) {
                    context.save();
                    // Flip by scaling and translating
                    context.scale(-1, 1);
                    const drawX = -(this.position.x - this.offset.x + scaledFrameWidth);
                    context.drawImage(
                        this.image,
                        this.frameCurrent * frameWidth,
                        0,
                        frameWidth,
                        imgHeight,
                        drawX,
                        this.position.y - this.offset.y,
                        scaledFrameWidth,
                        scaledHeight
                    );
                    context.restore();
                } else {
                    try {
                        context.drawImage(
                            this.image,
                            this.frameCurrent * frameWidth,
                            0,
                            frameWidth,
                            imgHeight,
                            this.position.x - this.offset.x,
                            this.position.y - this.offset.y,
                            scaledFrameWidth,
                            scaledHeight
                        )
                    } catch (e) {
                        // Image might not be fully loaded, that's okay
                    }
                }
            }
        }
    };

    animateFrames() {
        this.framesElapsed++;

        if (this.framesElapsed % this.framesHold === 0) {
            if (this.frameCurrent < this.framesMax - 1) {
                this.frameCurrent++;
            } else {
                this.frameCurrent = 0;
            }
        }
    }

    update() {
        this.draw();
        this.animateFrames()
    };
};

class Fighter extends Sprite {
    constructor({
        position,
        velocity,
        color = 'red',
        imageSrc,
        scale = 1,
        framesMax = 1,
        offset = { x: 0, y: 0 },
        sprites,
        attackBox = { offset: {}, width: undefined, height: undefined },
        health = 100,
        attack = 50,
        defense = 20,
        name = 'Unknown',
        flip = false  // If true, flip sprite horizontally
    }) {
        super({
            position,
            imageSrc,
            scale,
            framesMax,
            offset,
            flip: flip  // Pass flip to parent Sprite class
        })

        this.velocity = velocity;
        this.width = 50;
        this.height = 150;
        this.lastkey;
        this.attackBox = {
            position: {
                x: this.position.x,
                y: this.position.y
            },
            offset: attackBox.offset,
            width: attackBox.width,
            height: attackBox.height
        }
        this.color = color;
        this.isAttacking;
        this.health = health;
        this.maxHealth = health;
        this.attackPower = attack; // Renamed to avoid conflict with attack() method
        this.defense = defense;
        this.name = name;
        this.frameCurrent = 0;
        this.framesElapsed = 0;
        this.framesHold = 6; // Faster animation playback
        this.sprites = sprites;
        this.dead = false;

        for (const sprite in this.sprites) {
            // Support both sprite sheets and individual image arrays
            if (sprites[sprite].images && Array.isArray(sprites[sprite].images)) {
                // Load array of individual images
                sprites[sprite].images = sprites[sprite].images.map(imgSrc => {
                    const img = new Image();
                    // Preload image to ensure it's ready
                    img.onload = function() {
                        // Image loaded successfully
                    };
                    img.onerror = function() {
                        console.warn('Failed to load image:', imgSrc);
                    };
                    img.src = imgSrc;
                    return img;
                });
                sprites[sprite].image = sprites[sprite].images[0]; // Use first image as default
            } else {
                // Use sprite sheet (original behavior)
                sprites[sprite].image = new Image();
                sprites[sprite].image.onload = function() {
                    // Image loaded successfully
                };
                sprites[sprite].image.src = sprites[sprite].imageSrc;
            }
        }
        
        // Ensure initial sprite is set and visible
        if (this.sprites.idle) {
            if (this.sprites.idle.images && this.sprites.idle.images.length > 0) {
                this.images = this.sprites.idle.images;
                this.image = this.sprites.idle.images[0];
            } else if (this.sprites.idle.image) {
                this.images = null;
                this.image = this.sprites.idle.image;
            }
        }
    }

    update() {
        this.draw();
        if (!this.dead) this.animateFrames()

        // attack boxes
        this.attackBox.position.x = this.position.x + this.attackBox.offset.x;
        this.attackBox.position.y = this.position.y + this.attackBox.offset.y;

        // draw attack box (temporarily enabled for debugging)
        if (this.isAttacking) {
            context.strokeStyle = this.color === 'red' ? 'red' : 'blue';
            context.lineWidth = 2;
            context.strokeRect(
                this.attackBox.position.x,
                this.attackBox.position.y,
                this.attackBox.width,
                this.attackBox.height
            )
        }

        this.position.x += this.velocity.x;
        this.position.y += this.velocity.y;

        //gravity function
        // Floor is at canvas.height - 82 = 494
        // Character image is 200px tall, drawn at position.y - offset.y
        // So we want: (position.y - offset.y) + 200 = 494
        // Therefore: position.y = 494 - 200 + offset.y = 294 + offset.y
        const floorY = canvas.height - 82; // 494
        const characterHeight = 200; // TARGET_HEIGHT
        const targetY = floorY - characterHeight + this.offset.y;
        
        if (this.position.y + this.height + this.velocity.y >= floorY) {
            this.velocity.y = 0;
            this.position.y = targetY;
        } else {
            this.velocity.y += gravity;
        }
    }
    attack() {
        this.switchSprite('attack1');
        this.isAttacking = true;
    }

    takeHit(damage = null) {
        // Calculate damage: attacker's attackPower - defender's defense (minimum 1 damage)
        if (damage === null) {
            damage = 35; // Default damage if not specified
        }
        
        const oldHealth = this.health;
        
        // Apply defense reduction (defense reduces damage, minimum 1 damage)
        // Defense reduces damage by a percentage (like Arena 1)
        const defenseReduction = Math.min(0.5, this.defense / 100); // Max 50% reduction
        const actualDamage = Math.max(1, Math.floor(damage * (1 - defenseReduction)));
        this.health = Math.max(0, this.health - actualDamage);

        console.log(`💥 ${this.name} takes hit:`, {
            damage: actualDamage,
            oldHealth: oldHealth,
            newHealth: this.health,
            defense: this.defense,
            defenseReduction: defenseReduction,
            willDie: this.health <= 0
        });

        if (this.health <= 0) {
            console.log(`💀 ${this.name} health is 0, switching to death sprite`);
            this.switchSprite('death');
            console.log(`💀 Death sprite switched, has image: ${this.image ? 'yes' : 'no'}, images: ${this.images ? this.images.length : 'none'}`);
        } else {
            this.switchSprite('takeHit');
        }
    }

    switchSprite(sprite) {
        console.log(`🔄 ${this.name} switchSprite called:`, {
            requestedSprite: sprite,
            currentSprite: this.image ? 'has image' : 'no image',
            currentImages: this.images ? this.images.length + ' images' : 'no images',
            frameCurrent: this.frameCurrent,
            dead: this.dead
        });
        
        // overriding all animations when dies
        const currentSpriteImages = this.images || (this.image ? [this.image] : []);
        const deathImages = this.sprites.death.images || (this.sprites.death.image ? [this.sprites.death.image] : []);
        if (currentSpriteImages[0] === deathImages[0] || this.image === this.sprites.death.image) {
            if (this.frameCurrent === this.sprites.death.framesMax - 1) {
                console.log(`💀 ${this.name} death animation complete, setting dead=true`);
                this.dead = true;
            }
            console.log(`⚠️ ${this.name} already in death sprite, skipping switch`);
            return
        }

        // overriding all animations with attack
        const attackImages = this.sprites.attack1.images || (this.sprites.attack1.image ? [this.sprites.attack1.image] : []);
        if ((currentSpriteImages[0] === attackImages[0] || this.image === this.sprites.attack1.image) &&
            this.frameCurrent < this.sprites.attack1.framesMax - 1)
            return
        // overriding all animations when take hit
        const takeHitImages = this.sprites.takeHit.images || (this.sprites.takeHit.image ? [this.sprites.takeHit.image] : []);
        if ((currentSpriteImages[0] === takeHitImages[0] || this.image === this.sprites.takeHit.image) &&
            this.frameCurrent < this.sprites.takeHit.framesMax - 1)
            return

        switch (sprite) {
            case 'idle':
                if (this.sprites.idle.images) {
                    this.images = this.sprites.idle.images;
                    this.image = this.sprites.idle.images[0];
                } else {
                    this.images = null;
                    this.image = this.sprites.idle.image;
                }
                this.framesMax = this.sprites.idle.framesMax;
                this.frameCurrent = 0;
                break;
            case 'run':
                if (this.sprites.run && this.sprites.run.images) {
                    this.images = this.sprites.run.images;
                    this.image = this.sprites.run.images[0];
                } else if (this.sprites.run) {
                    this.images = null;
                    this.image = this.sprites.run.image;
                }
                if (this.sprites.run) {
                    this.framesMax = this.sprites.run.framesMax;
                    this.frameCurrent = 0;
                }
                break;
            case 'jump':
                if (this.sprites.jump && this.sprites.jump.images) {
                    this.images = this.sprites.jump.images;
                    this.image = this.sprites.jump.images[0];
                } else if (this.sprites.jump) {
                    this.images = null;
                    this.image = this.sprites.jump.image;
                }
                if (this.sprites.jump) {
                    this.framesMax = this.sprites.jump.framesMax;
                    this.frameCurrent = 0;
                }
                break;
            case 'fall':
                if (this.sprites.fall && this.sprites.fall.images) {
                    this.images = this.sprites.fall.images;
                    this.image = this.sprites.fall.images[0];
                } else if (this.sprites.fall) {
                    this.images = null;
                    this.image = this.sprites.fall.image;
                }
                if (this.sprites.fall) {
                    this.framesMax = this.sprites.fall.framesMax;
                    this.frameCurrent = 0;
                }
                break;
            case 'attack1':
                if (this.sprites.attack1.images) {
                    this.images = this.sprites.attack1.images;
                    this.image = this.sprites.attack1.images[0];
                } else {
                    this.images = null;
                    this.image = this.sprites.attack1.image;
                }
                this.framesMax = this.sprites.attack1.framesMax;
                this.frameCurrent = 0;
                break;
            case 'takeHit':
                if (this.sprites.takeHit.images) {
                    this.images = this.sprites.takeHit.images;
                    this.image = this.sprites.takeHit.images[0];
                } else {
                    this.images = null;
                    this.image = this.sprites.takeHit.image;
                }
                this.framesMax = this.sprites.takeHit.framesMax;
                this.frameCurrent = 0;
                break;
            case 'death':
                console.log(`💀 ${this.name} switching to death sprite:`, {
                    hasDeathImages: !!this.sprites.death.images,
                    deathImagesCount: this.sprites.death.images ? this.sprites.death.images.length : 0,
                    hasDeathImage: !!this.sprites.death.image,
                    framesMax: this.sprites.death.framesMax
                });
                if (this.sprites.death.images) {
                    this.images = this.sprites.death.images;
                    this.image = this.sprites.death.images[0];
                    console.log(`✅ ${this.name} death sprite set from images array (${this.sprites.death.images.length} images), first image: ${this.sprites.death.images[0]}`);
                } else {
                    this.images = null;
                    this.image = this.sprites.death.image;
                    console.log(`✅ ${this.name} death sprite set from single image: ${this.sprites.death.image}`);
                }
                this.framesMax = this.sprites.death.framesMax;
                this.frameCurrent = 0;
                console.log(`✅ ${this.name} death sprite initialized: framesMax=${this.framesMax}, frameCurrent=${this.frameCurrent}, hasImage=${!!this.image}`);
                break;
            case 'win':
                console.log(`🏆 ${this.name} switching to win sprite:`, {
                    hasWinImages: !!this.sprites.win.images,
                    winImagesCount: this.sprites.win.images ? this.sprites.win.images.length : 0,
                    hasWinImage: !!this.sprites.win.image,
                    framesMax: this.sprites.win.framesMax
                });
                if (this.sprites.win && this.sprites.win.images) {
                    this.images = this.sprites.win.images;
                    this.image = this.sprites.win.images[0];
                    console.log(`✅ ${this.name} win sprite set from images array (${this.sprites.win.images.length} images), first image: ${this.sprites.win.images[0]}`);
                } else if (this.sprites.win) {
                    this.images = null;
                    this.image = this.sprites.win.image;
                    console.log(`✅ ${this.name} win sprite set from single image: ${this.sprites.win.image}`);
                }
                if (this.sprites.win) {
                    this.framesMax = this.sprites.win.framesMax;
                    this.frameCurrent = 0;
                    console.log(`✅ ${this.name} win sprite initialized: framesMax=${this.framesMax}, frameCurrent=${this.frameCurrent}, hasImage=${!!this.image}`);
                }
                break;
        }
    }
};
